/**
* \file: GstreamerVideoPipeline.cpp
*
* \version: 1.0
*
* \release: $Name:$
*
* Configuration of GstreamerVideoSink.
*
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
************************************************************************/

#include "GstreamerVideoPipeline.h"
#include <adit_logging.h>
#include <gst/app/gstappsrc.h>

LOG_IMPORT_CONTEXT(bdcl_gst)

namespace adit { namespace bdcl {

static GMainLoop* messageBusLoop = nullptr;
static int messageBusRef = 0;
static pthread_mutex_t messageBusMutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_t messageBusThread = 0;

static void* mainLoop(void* inData);

GstreamerVideoPipeline::GstreamerVideoPipeline()
{
    pipeline = nullptr;
    watchId = 0;
    bus = nullptr;
    partial = nullptr;
    appsrc = nullptr;
    gstVideoSink = nullptr;
    MessageBusAddRef();

}
GstreamerVideoPipeline::~GstreamerVideoPipeline()
{

}

bool GstreamerVideoPipeline::CreatePipeline(const std::string& inAppSrcName, const std::string& inPartialBinName,
        const std::string& inPipeline)
{
    LOGD_DEBUG((bdcl_gst, "video out pipeline: appsrc ! %s", inPipeline.c_str()));

    // creates 'pipeline' with 'partial' bin inside
    if (!CreatePartialBin(inPartialBinName, inPipeline))
    {
        // error logged
        return false;
    }

    if (nullptr == (appsrc = gst_element_factory_make("appsrc", inAppSrcName.c_str())))
    {
        LOG_ERROR((bdcl_gst, "could not create appsrc %s", inAppSrcName.c_str()));
        return false; /* ========== leaving function ========== */
    }

    gst_app_src_set_latency(GST_APP_SRC(appsrc), 0, 100000); // TODO performance: configuration? what is appropriate?

    if (!gst_bin_add(GST_BIN(pipeline), appsrc))
    {
        LOG_ERROR((bdcl_gst, "could not add appsrc element %s", inAppSrcName.c_str()));
        return false; /* ========== leaving function ========== */
    }

    if (!gst_element_link(appsrc, partial))
    {
        LOG_ERROR((bdcl_gst, "could not link appsrc %s", inAppSrcName.c_str()));
        return false; /* ========== leaving function ========== */
    }

    return true;
}

bool GstreamerVideoPipeline::CreatePartialBin(const std::string& inName, const std::string& inPipeline)
{
    GError* error = nullptr;

    partial = gst_parse_bin_from_description(inPipeline.c_str(), TRUE, &error);

    if (error != nullptr)
    {
        LOG_ERROR((bdcl_gst, "gst_parse_bin_from_description: %d %s", error->code, error->message));
        g_error_free(error);
        return false; /* ========== leaving function ========== */
    }

    LOGD_VERBOSE((bdcl_gst, "pipeline created"));

    // create pipeline bin
    pipeline = gst_pipeline_new(inName.c_str());
    if (pipeline == nullptr)
    {
        LOG_ERROR((bdcl_gst, "gst_pipeline_new failed"));
        return false; /* ========== leaving function ========== */
    }

    // add bus callback
    bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));
    if (bus != nullptr)
    {
        watchId = gst_bus_add_watch(bus, (GstBusFunc)busCallBack, (void*)this);
    }
    else
    {
        LOG_ERROR((bdcl_gst, "gst_pipeline_get_bus failed"));
        return false; /* ========== leaving function ========== */
    }

    gst_object_ref(partial); // pipeline takes ownership of partial, retain reference
    if (!gst_bin_add(GST_BIN(pipeline), partial))
    {
        gst_object_unref(partial); // failed, so release partial
        LOG_ERROR((bdcl_gst, "could not add configured part to the pipeline"));
        return false; /* ========== leaving function ========== */
    }

    GstIterator* iter = gst_bin_iterate_sinks(GST_BIN(partial));
    if (iter == nullptr)
    {
        LOG_ERROR((bdcl_gst, "Could not get the iterator from GStreamer for the sinks"));
        return false;
    }
    return true;
}

gboolean busCallBack(GstBus* inBus, GstMessage* inMessage, gpointer inPointer)
{
    (void)inBus;

    auto me = static_cast<GstreamerVideoPipeline*>(inPointer);

    switch (GST_MESSAGE_TYPE(inMessage))
    {
    // GST_EOS: may be ignored verbosely
    case GST_MESSAGE_ERROR:
        {
            gchar* debug;
            GError* err;
            gst_message_parse_error(inMessage, &err, &debug);
            g_error_free(err);
            g_free(debug);

            pthread_mutex_lock(&me->stopConditionMutex);
            me->stopped = true;
            pthread_cond_signal(&me->stopCondition);
            pthread_mutex_unlock(&me->stopConditionMutex);
        }
        break;
    default:
        break;
    }

    return TRUE;
}

void GstreamerVideoPipeline::StopPipeline()
{
    stopped = true;

    if (pipeline != nullptr)
    {
        auto ret = gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_NULL);
        if (ret == GST_STATE_CHANGE_FAILURE)
        {
            // silent warning
            LOGD_DEBUG((bdcl_gst, "gst_element_set_state GST_STATE_NULL failed"));
        }

        gst_element_get_state(GST_ELEMENT(pipeline), NULL, NULL, 1);

        gst_object_unref(pipeline);
        pipeline = nullptr;
    }

    // remove bus callback
    if (watchId != 0)
    {
        g_source_remove(watchId);
        watchId = 0;
    }

    if (bus != nullptr)
    {
        gst_object_unref(bus);
        bus = nullptr;
    }

    if (partial != nullptr)
    {
        gst_object_unref(partial);
        partial = nullptr;
    }
}

void MessageBusAddRef()
{

    // one static shared main loop for the message bus
    pthread_mutex_lock(&messageBusMutex);

    messageBusRef ++;
    if (messageBusRef == 1)
    {
        messageBusLoop = g_main_loop_new(nullptr, FALSE);
        if (messageBusLoop == nullptr)
            LOG_ERROR((bdcl_gst, "g_main_loop_new failed"));

        if (0 != pthread_create(&messageBusThread, nullptr, mainLoop, nullptr))
        {
            messageBusThread = 0;
            LOG_ERROR((bdcl_gst, "could not create Gstreamer message bus thread"));
        }
    }

    pthread_mutex_unlock(&messageBusMutex);
}

void MessageBusUnref()
{
    // one static shared main loop for the message bus
    pthread_mutex_lock(&messageBusMutex);

    messageBusRef --;
    if (messageBusRef == 0)
    {
        if (messageBusLoop != 0)
        {
            // TODO make sure the main loop is already started
            g_main_loop_quit(messageBusLoop);
            g_main_loop_unref(messageBusLoop);
            messageBusLoop = nullptr;
        }

        if (messageBusThread != 0)
        {
            pthread_join(messageBusThread, nullptr);
            messageBusThread = 0;
        }
    }

    pthread_mutex_unlock(&messageBusMutex);
}

void* mainLoop(void* inData)
{
   (void)inData;

   // set thread name
    prctl(PR_SET_NAME, "GstMainLoop", 0, 0, 0);

    g_main_loop_run(messageBusLoop);
    return nullptr;
}
} } /*namespace adit { namespace bdcl { */

